using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using JetBrains.Application;
using JetBrains.ReSharper.Feature.Services.Refactorings;
using JetBrains.ReSharper.Feature.Services.Refactorings.Conflicts;
using JetBrains.ReSharper.Psi.Resolve;
using ReSharperPlugin.Refix.Extensions;
using ReSharperPlugin.Refix.Fixes;

namespace ReSharperPlugin.Refix.Components;

[ShellComponent]
public class FixesManager : IFixesManager
{
    private readonly Dictionary<Type, List<FixDescriptor>> _availableFixes = new();

    public FixesManager(IMessaging messaging)
    {
        var assembly = Assembly.GetAssembly(typeof(FixesManager));
        foreach (var preFixType in assembly.GetTypes().Where(x => x.GetCustomAttributes<PreFixAttribute>().Any()))
            TryInsertFixIntoDict(preFixType, FixKind.PreFix);
        foreach (var postFixType in assembly.GetTypes().Where(x => x.GetCustomAttributes<PostFixAttribute>().Any()))
            TryInsertFixIntoDict(postFixType, FixKind.PostFix);
        messaging.ShowInitialNotification(_availableFixes.SelectMany(x => x.Value).Select(x => x.Kind).ToList());
    }

    public ISet<IReference> GetFixableReferences(IRefactoringWorkflow workflow, ConflictSearchResult conflictSearchResult)
    {
        workflow = workflow.Unwrapped();
        var conflicts = new HashSet<IReference>();
        if (!_availableFixes.TryGetValue(workflow.GetType(), out var fixList))
            return conflicts;
        foreach (var fixDescriptor in fixList)
            try
            {
                var fix = fixDescriptor.Create(workflow);
                conflicts.UnionWith(fix?.GetFixableReferences(conflictSearchResult) ?? conflicts);
            }
            catch (Exception e)
            {
                Logging.LogException(e);
            }
        return conflicts;
    }

    public int RunFixes(FixKind kind, IRefactoringWorkflow workflow, ConflictSearchResult conflictSearchResult)
    {
        workflow = workflow.Unwrapped();
        var fixedIssues = 0;
        if (!_availableFixes.TryGetValue(workflow.GetType(), out var fixList))
            return fixedIssues;
        foreach (var fixDescriptor in fixList.Where(x => x.Kind == kind))
            try
            {
                var fix = fixDescriptor.Create(workflow);
                fixedIssues += fix?.Execute(conflictSearchResult) ?? 0;
            }
            catch (Exception e)
            {
                Logging.LogException(e);
            }
        return fixedIssues;
    }

    private bool TryInsertFixIntoDict(Type fixType, FixKind fixKind)
    {
        if (!fixType.TryGetWorkflowType(out var workflowType))
        {
            Messaging.ShowFixRegistrationError(fixType.FullName);
            return false;
        }
        if (!fixType.TryMakeCreationFunc(out var creationFunc))
        {
            Messaging.ShowFixRegistrationError(fixType.FullName);
            return false;
        }
        var descriptor = new FixDescriptor { Kind = fixKind, Create = creationFunc };
        if (_availableFixes.TryGetValue(workflowType, out var fixList))
            fixList.Add(descriptor);
        else
            _availableFixes[workflowType] = new List<FixDescriptor> { descriptor };
        return true;
    }
}
