using System.Collections.Generic;
using System.Linq;
using JetBrains.Application;
using JetBrains.Application.Notifications;
using JetBrains.Diagnostics;
using JetBrains.Lifetimes;
using JetBrains.Util;
using JetBrains.Util.Logging;
using ReSharperPlugin.Refix.Fixes;

namespace ReSharperPlugin.Refix.Components;

[ShellComponent]
public class Messaging : IMessaging
{
    private const string DefaultCaption = $"{PluginData.Name} plugin";

    private readonly Lifetime _lifetime;
    private readonly UserNotifications _userNotifications;

    public Messaging(Lifetime lifetime, UserNotifications userNotifications)
    {
        _lifetime = lifetime;
        _userNotifications = userNotifications;
    }

    private static string GetPluralSuffix(int count, bool includeE = false) => count != 1
        ? (includeE ? "es" : "s")
        : "";

    public void ShowInitialNotification(List<FixKind> loadedFixes)
    {
        var preFixCount = loadedFixes.Count(x => x == FixKind.PreFix);
        var postFixCount = loadedFixes.Count(x => x == FixKind.PostFix);
        var title = $"Initialized {PluginData.Name} v{PluginData.Version}";
        var message = $"with {preFixCount} pre-fix{GetPluralSuffix(preFixCount, true)} " +
            $"and {postFixCount} post-fix{GetPluralSuffix(postFixCount, true)} loaded";

        Logger.LogMessage(LoggingLevel.INFO, $"{title} {message}");

        _userNotifications.CreateNotification(
            _lifetime,
            NotificationSeverity.INFO,
            title: title,
            body: message,
            closeAfterExecution: true);
    }

    public void ShowFixedNotification(string refactoringName, int fixedIssuesCount)
    {
        if (fixedIssuesCount <= 0)
            return;

        var title = $"Fixed {fixedIssuesCount} issue{GetPluralSuffix(fixedIssuesCount)} in {refactoringName} refactoring";

        Logger.LogMessage(LoggingLevel.INFO, title);

        _userNotifications.CreateNotification(
            _lifetime,
            NotificationSeverity.INFO,
            title: title,
            body: DefaultCaption,
            closeAfterExecution: true);
    }

    public static void ShowUnsupportedWarning(string refactoringTitle, string reason)
    {
        MessageBox.ShowExclamation($"Unsupported feature in \"{refactoringTitle}\" refactoring: {reason}",
            DefaultCaption);
    }

    public static void ShowUnsupportedError(string refactoringTitle, string reason)
    {
        MessageBox.ShowError($"Cannot fix \"{refactoringTitle}\" refactoring: {reason}", DefaultCaption);
    }

    public static void ShowFixRegistrationError(string fixName)
    {
        MessageBox.ShowError($"Error registering fix of type {fixName}");
    }

    public static void ShowUsingAdded(string namespaceName, string elementName)
    {
        MessageBox.ShowInfo($"Added namespace {namespaceName} to file containing {elementName}");
    }
}
