using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;
using JetBrains.Diagnostics;
using JetBrains.ProjectModel;
using JetBrains.ReSharper.Feature.Services.Refactorings;
using JetBrains.ReSharper.Feature.Services.Refactorings.Conflicts;
using JetBrains.ReSharper.Psi.Resolve;
using JetBrains.ReSharper.Resources.Shell;
using ReSharperPlugin.Refix.Extensions;

namespace ReSharperPlugin.Refix.Fixes;

public abstract class Fix<TFix, TWorkflow> : IFix
    where TFix : Fix<TFix, TWorkflow>, new()
    where TWorkflow : IRefactoringWorkflow
{
    protected ISolution Solution { get; private set; }

    [CanBeNull]
    public static IFix Create([NotNull] TWorkflow workflow)
    {
        Assertion.Assert(workflow is not null);
        var instance = new TFix();
        return instance.Initialize(workflow) ? instance : null;
    }

    protected virtual bool Initialize([NotNull] TWorkflow workflow)
    {
        Solution = workflow.Solution;
        return true;
    }

    public ISet<IReference> GetFixableReferences(ConflictSearchResult conflictSearchResult)
    {
        using var readCookie = ReadLockCookie.Create();

        var allReferences = conflictSearchResult.Conflicts.ToOccurrences(Solution).ToReferences();
        return allReferences.Where(IsApplicableTo).ToHashSet();
    }

    protected abstract bool IsApplicableTo(IReference reference);

    public abstract int Execute(ConflictSearchResult conflictSearchResult);
}
