using System.Linq;
using JetBrains.Diagnostics;
using JetBrains.ReSharper.Feature.Services.Refactorings.Conflicts;
using JetBrains.ReSharper.Psi;
using JetBrains.ReSharper.Psi.CodeStyle;
using JetBrains.ReSharper.Psi.CSharp;
using JetBrains.ReSharper.Psi.CSharp.Tree;
using JetBrains.ReSharper.Psi.ExtensionsAPI;
using JetBrains.ReSharper.Psi.ExtensionsAPI.Tree;
using JetBrains.ReSharper.Psi.Resolve;
using JetBrains.ReSharper.Psi.Transactions;
using JetBrains.ReSharper.Psi.Tree;
using JetBrains.ReSharper.Refactorings.Convert.Property2Function;
using JetBrains.ReSharper.Resources.Shell;
using JetBrains.Util;
using JetBrains.Util.Logging;
using ReSharperPlugin.Refix.Extensions;

namespace ReSharperPlugin.Refix.Fixes;

[PreFix]
public class RefixProperty2Function : Fix<RefixProperty2Function, Property2FunctionWorkflow>
{
    public const string Title = "Convert Property to Method(s)";

    protected override bool Initialize(Property2FunctionWorkflow workflow)
    {
        base.Initialize(workflow);

        Assertion.Assert(workflow is not null);
        if (!workflow.DataProvider.ConvertSetter)
        {
            Logger.LogMessage(LoggingLevel.TRACE, $"{nameof(RefixProperty2Function)}: ConvertSetter not checked, skipping");
            return false;
        }

        return true;
    }

    protected override bool IsApplicableTo(IReference reference)
    {
        return reference.GetTreeNode() is IPropertyInitializer;
    }

    public override int Execute(ConflictSearchResult conflictSearchResult)
    {
        var psiServices = Solution.GetPsiServices();
        using var psiCookie = new PsiTransactionCookie(psiServices, DefaultAction.Rollback, $"{PluginData.Name}: {Title}");
        using var readCookie = ReadLockCookie.Create();

        var conflicts = conflictSearchResult.Conflicts.ToList();
        var occurrences = conflicts.ToOccurrences(Solution).ToList();
        var references = occurrences.ToReferences().ToList();
        var nodes = references.Select(x => x.GetTreeNode());

        var fixedNodes = 0;

        foreach (var node in nodes)
        {
            if (node is not IPropertyInitializer propertyInitializer)
                continue;

            var objectInitializer = propertyInitializer.FirstAncestorOfType<IObjectInitializer>();
            Assertion.Assert(objectInitializer != null);
            var objectCreationExpression = objectInitializer.FirstAncestorOfType<IObjectCreationExpression>();
            Assertion.Assert(objectCreationExpression != null);
            var objectCreationParent = objectCreationExpression.Parent;
            Assertion.Assert(objectCreationParent != null);

            var factory = CSharpElementFactory.GetInstance(propertyInitializer);

            using var formatterCookie = new DisableCodeFormatter();
            using var writeCookie = WriteLockCookie.Create();

            const string variableName = "ret";

            var propertyInitializerCopy = propertyInitializer.Copy();
            objectInitializer.RemoveMemberInitializer(propertyInitializer);

            var objectCreationCopy = objectCreationExpression.Copy();
            objectCreationCopy.SetTypeUsage(factory.CreateTypeUsage(objectCreationExpression.Type(), objectCreationExpression));

            var lambdaCall = factory.CreateExpression(
                "((System.Func<$0>)(() => { var " + variableName + " = $1; " + variableName + ".$2 = $3; return " + variableName + "; }))()",
                objectCreationExpression.Type(),
                objectCreationCopy,
                propertyInitializerCopy.NameIdentifier,
                propertyInitializerCopy.Expression);

            ModificationUtil.ReplaceChild(objectCreationExpression, lambdaCall);

            objectCreationParent.FormatNode();
            fixedNodes++;
        }
        psiCookie.Commit();
        return fixedNodes;
    }
}
