using System.Collections.Generic;
using System.Linq;
using JetBrains.Diagnostics;
using JetBrains.ReSharper.Feature.Services.Refactorings.Conflicts;
using JetBrains.ReSharper.Psi;
using JetBrains.ReSharper.Psi.CSharp;
using JetBrains.ReSharper.Psi.CSharp.Impl;
using JetBrains.ReSharper.Psi.CSharp.Tree;
using JetBrains.ReSharper.Psi.ExtensionsAPI;
using JetBrains.ReSharper.Psi.Resolve;
using JetBrains.ReSharper.Psi.Transactions;
using JetBrains.ReSharper.Psi.Tree;
using JetBrains.ReSharper.Psi.Util;
using JetBrains.ReSharper.Refactorings.MoveMembers.ImplPushDown;
using JetBrains.ReSharper.Refactorings.Util;
using JetBrains.ReSharper.Resources.Shell;
using JetBrains.Util;
using ReSharperPlugin.Refix.Components;
using ReSharperPlugin.Refix.Extensions;

namespace ReSharperPlugin.Refix.Fixes;

[PostFix]
public class RefixPushDown : Fix<RefixPushDown, PushDownWorkflow>
{
    public const string Title = "Push Members Down";

    private List<ITypeDeclaration> _selectedInheritors;

    protected override bool Initialize(PushDownWorkflow workflow)
    {
        base.Initialize(workflow);

        Assertion.Assert(workflow is not null);
        var selectedInheritors = workflow.DataModel.SelectedInheritorsItems.Select(x => x.TypeDeclaration).ToList();

        _selectedInheritors = selectedInheritors;

        return true;
    }

    protected override bool IsApplicableTo(IReference reference)
    {
        var declaredElement = reference.Resolve().DeclaredElement;
        if (declaredElement is not ILocalVariableDeclaration localVariableDeclaration)
            return false;
        var accessibleTypes = GetAccessibleInheritors(localVariableDeclaration);
        if (accessibleTypes.Count == 0)
            return false;
        return GetTypesSortedConstructorLength(localVariableDeclaration, accessibleTypes).Count != 0;
    }

    public override int Execute(ConflictSearchResult conflictSearchResult)
    {
        var psiServices = Solution.GetPsiServices();
        using var psiCookie = new PsiTransactionCookie(psiServices, DefaultAction.Rollback, $"{PluginData.Name}: {Title}");
        using var readCookie = ReadLockCookie.Create();

        var conflicts = conflictSearchResult.Conflicts.ToList();
        var references = conflicts.ToOccurrences(Solution).ToReferences().ToList();
        var resolvedReferences = references.Select(x => x.Resolve().DeclaredElement).Where(x => x != null).Distinct()
            .ToList();

        var fixedReferences = 0;

        using var formatterCookie = new DisableCodeFormatter();
        using var writeCookie = WriteLockCookie.Create();
        foreach (var reference in resolvedReferences)
        {
            if (reference is ILocalVariableDeclaration variableDeclaration)
            {
                var factory = CSharpElementFactory.GetInstance(variableDeclaration);
                var accessibleTypes = GetAccessibleInheritors(variableDeclaration);
                if (accessibleTypes.Count == 0)
                {
                    Messaging.ShowUnsupportedError(Title, $"No inherited type accessible from {variableDeclaration} context");
                    continue;
                }
                var typesSortedByConstructorLength = GetTypesSortedConstructorLength(variableDeclaration, accessibleTypes);
                if (typesSortedByConstructorLength.Count == 0)
                {
                    Messaging.ShowUnsupportedError(Title, $"No compatible constructor accessible from {variableDeclaration} context");
                    continue;
                }
                var bestCombination = typesSortedByConstructorLength.FirstOrDefault();
                var namespaceToImport = bestCombination.Type.GetContainingNamespace();
                var currentNamespace = variableDeclaration.GetContainingNamespaceDeclaration();
                if (namespaceToImport.QualifiedName != "" && !UsingUtil.CheckNamespaceAlreadyImported(currentNamespace, namespaceToImport))
                {
                    UsingUtil.AddImportTo(currentNamespace, namespaceToImport);
                    Messaging.ShowUsingAdded(namespaceToImport.QualifiedName, $"variable {variableDeclaration.NameIdentifier.Name}");
                }
                var typeUsage = factory.CreateTypeUsage("$0", bestCombination.Type);
                var initializer = factory.CreateExpressionInitializer(factory.CreateExpression("new $0()", typeUsage));
                variableDeclaration.SetInitial(initializer);
                if (!variableDeclaration.IsVar)
                    variableDeclaration.SetTypeUsage(typeUsage);
                fixedReferences++;
            }
        }
        psiCookie.Commit();
        return fixedReferences;
    }

    private IList<ITypeElement> GetAccessibleInheritors(ITreeNode accessPoint)
    {
        var inheritingTypes = _selectedInheritors.Select(x => x.DeclaredElement).ToList();
        return inheritingTypes.Where(x => VisibilityUtil.TargetTypeAccessible(accessPoint, x)).ToList();
    }

    private static IList<(ITypeElement Type, IConstructor BestConstructor)> GetTypesSortedConstructorLength(
        ITreeNode accessPoint, IEnumerable<ITypeElement> accessibleTypes)
    {
        IConstructor BestConstructor(ITypeElement typeElement) => typeElement.Constructors
            .OrderBy(x => x.Parameters.Count)
            .FirstOrDefault(x => AccessUtil.IsSymbolAccessible(x, new ElementAccessContext(accessPoint)));
        return accessibleTypes
            .Select(x => (Type: x, BestConstructor: BestConstructor(x)))
            .OrderBy(x => x.BestConstructor.Parameters.Count).ToList();
    }
}
