using System.Linq;
using JetBrains.Diagnostics;
using JetBrains.ReSharper.Feature.Services.Refactorings.Conflicts;
using JetBrains.ReSharper.Psi;
using JetBrains.ReSharper.Psi.CSharp;
using JetBrains.ReSharper.Psi.CSharp.Tree;
using JetBrains.ReSharper.Psi.ExtensionsAPI;
using JetBrains.ReSharper.Psi.Resolve;
using JetBrains.ReSharper.Psi.Transactions;
using JetBrains.ReSharper.Refactorings.TransformParameters;
using JetBrains.ReSharper.Resources.Shell;
using JetBrains.Util;
using ReSharperPlugin.Refix.Components;

namespace ReSharperPlugin.Refix.Fixes;

[PostFix]
public class RefixTransformParameters : Fix<RefixTransformParameters, TransformParametersWorkflow>
{
    public const string Title = "Transform Parameters";
    private CrucialDataModel _data;

    protected override bool Initialize(TransformParametersWorkflow workflow)
    {
        base.Initialize(workflow);

        Assertion.Assert(workflow is not null);
        var model = new TransformParametersDataModel(workflow.ViewModel);
        Assertion.Assert(model.ParametersOwner is not null);

        using (ReadLockCookie.Create())
            if (!model.ParametersOwner.GetDeclarations().OfType<IExpressionBodyOwnerDeclaration>().Any())
                return false;  // not this case

        var outTransform = model.OutInfo.OutEntityType;
        if (outTransform is ReturnTypeEnum.NONE or ReturnTypeEnum.CAN_NOT_BE_CHANGED)
        {
            Messaging.ShowUnsupportedWarning(Title, "no 'out' parameter transformed");
            return false;
        }

        if (model.OutInfo.ReturnParameterInfo?.Included is true)
        {
            Messaging.ShowUnsupportedWarning(Title, "more than one value used as return");
            return false;
        }

        var includedParameters = model.OutInfo.Parameters.Where(x => x.Included).ToList();
        if (includedParameters.Count != 1)
        {
            Messaging.ShowUnsupportedWarning(Title, includedParameters.Count == 0
                ? "no 'out' parameter transformed"
                : "more than one value used as return");
            return false;
        }

        _data = new CrucialDataModel
        {
            ParametersOwner = model.ParametersOwner,
            ParameterName = includedParameters[0].Name,
            OutEntityType = outTransform,
            TargetClassName = model.OutInfo.TargetClassName
        };

        return true;
    }

    protected override bool IsApplicableTo(IReference reference)
    {
        return false;
    }

    public override int Execute(ConflictSearchResult conflictSearchResult)
    {
        var psiServices = Solution.GetPsiServices();
        using var psiCookie = new PsiTransactionCookie(psiServices, DefaultAction.Rollback, $"{PluginData.Name}: {Title}");
        using var readCookie = ReadLockCookie.Create();

        var expressionBodyOwner = _data.ParametersOwner.GetDeclarations()
            .OfType<IExpressionBodyOwnerDeclaration>().First();

        var arrowClause = expressionBodyOwner.ArrowClause;
        var expression = arrowClause.Expression as IAssignmentExpression;
        Assertion.Assert(expression is not null);

        var destination = expression.Dest as IReferenceExpression;
        Assertion.Assert(destination is not null);

        Assertion.Assert(destination.NameIdentifier.Name == _data.ParameterName,
            $"{destination.NameIdentifier.Name} not equal to {_data.ParameterName}");

        var factory = CSharpElementFactory.GetInstance(expressionBodyOwner);

        using var formatterCookie = new DisableCodeFormatter();
        using var writeCookie = WriteLockCookie.Create();
        var newExpression = expression.Source;
        if (_data.OutEntityType is ReturnTypeEnum.CLASS_FIELDS or ReturnTypeEnum.CLASS_PROPERTIES)
            newExpression = factory.CreateExpression("new $0($1)",
                _data.TargetClassName,
                newExpression) as IObjectCreationExpression;
        arrowClause.SetExpression(newExpression);
        psiCookie.Commit();
        return 1;
    }

    private struct CrucialDataModel
    {
        public IParametersOwner ParametersOwner { get; init; }
        public string ParameterName { get; init; }
        public ReturnTypeEnum OutEntityType { get; init; }
        public string TargetClassName { get; init; }
    }
}
