using System.Collections.Generic;
using System.Linq;
using JetBrains.Application.Progress;
using JetBrains.Application.UI.Controls.JetPopupMenu.Detail;
using JetBrains.ReSharper.Feature.Services.Refactorings;
using JetBrains.ReSharper.Feature.Services.Refactorings.Conflicts;
using ReSharperPlugin.Refix.Components;
using ReSharperPlugin.Refix.Extensions;

namespace ReSharperPlugin.Refix.Wrappers;

public class ConflictSearcherWrapper : IConflictSearcher
{
    public IConflictSearcher WrappedSearcher { get; }

    private readonly IRefactoringWorkflow _workflow;
    private readonly IFixesManager _fixesManager;
    private readonly IMessaging _messaging;

    public ConflictSearcherWrapper(IConflictSearcher wrappedSearcher, IRefactoringWorkflow workflow, IFixesManager fixesManager, IMessaging messaging) =>
        (WrappedSearcher, _workflow, _fixesManager, _messaging) = (wrappedSearcher, workflow, fixesManager, messaging);

    public ConflictSearchResult SearchConflicts(IProgressIndicator progressIndicator, bool canPerformRefactoring)
    {
        Logging.LogCallerData();
        var fixedIssues = 0;
        var resultBeforePreFixes = SearchConflictsWithoutConsequences();
        var resultBeforePreFixesWithFixableMarked = CreateResultWithFixableMarked(resultBeforePreFixes);
        if (canPerformRefactoring)
            fixedIssues += _fixesManager.RunPreFixes(_workflow, resultBeforePreFixes);

        var result = WrappedSearcher.SearchConflicts(progressIndicator, canPerformRefactoring);
        var resultWithFixableMarked = CreateResultWithFixableMarked(result);

        if (!result.PerformedRefactoring)
            return resultBeforePreFixesWithFixableMarked;

        fixedIssues += _fixesManager.RunPostFixes(_workflow, result);
        if (fixedIssues > 0)
            _messaging.ShowFixedNotification(MnemonicStore.RemoveMnemonicMark(_workflow.Title), fixedIssues);
        return resultWithFixableMarked;
    }

    public ConflictSearchResult SearchConflictsWithoutConsequences() =>
        WrappedSearcher.SearchConflicts(NullProgressIndicator.Create(), false);

    private ConflictSearchResult CreateResultWithFixableMarked(ConflictSearchResult originalResult)
    {
        const string infoConflictDescription = $"ⓘ {{0,The conflict}} can be automatically fixed by {PluginData.Name}";

        if (originalResult.Failed || originalResult.PerformedRefactoring)
            return originalResult;

        var originalConflicts = new List<IConflict>(originalResult.Conflicts);
        var processedConflicts = new List<IConflict>();
        var fixableReferences = _fixesManager.GetFixableReferences(_workflow, new ConflictSearchResult(originalConflicts));
        foreach (var conflict in originalConflicts)
        {
            processedConflicts.Add(conflict);
            processedConflicts.AddRange(conflict.ToOccurrences(_workflow.Solution)
                .ToReferences()
                .Where(fixableReferences.Contains)
                .Select(reference => Conflict.Create(reference, infoConflictDescription, ConflictSeverity.Warning)));
        }
        return new ConflictSearchResult(processedConflicts);
    }

    #region Pass-through properties

    public ConflictSearchResult LastResult => WrappedSearcher.LastResult;

    #endregion Pass-through properties
}
