using System;
using JetBrains.Annotations;
using JetBrains.Application.DataContext;
using JetBrains.Application.Progress;
using JetBrains.Application.UI.Components;
using JetBrains.Application.UI.Controls.JetPopupMenu.Detail;
using JetBrains.Lifetimes;
using JetBrains.ProjectModel;
using JetBrains.ReSharper.Feature.Services.Refactorings;
using JetBrains.ReSharper.Feature.Services.Refactorings.Conflicts;
using JetBrains.UI.Icons;
using JetBrains.Util.EventBus;
using ReSharperPlugin.Refix.Components;
using ReSharperPlugin.Refix.Extensions;

namespace ReSharperPlugin.Refix.Wrappers;

internal class RefactoringWorkflowWrapper : IRefactoringWorkflow
{
    public IRefactoringWorkflow WrappedWorkflow { get; }
    [CanBeNull] public ConflictSearcherWrapper ConflictSearcherWrapper { get; }
    public IConflictSearcher ConflictSearcher => ConflictSearcherWrapper;

    private readonly IFixesManager _fixesManager;
    private readonly IMessaging _messaging;

    public RefactoringWorkflowWrapper(IRefactoringWorkflow wrappedWorkflow, IFixesManager fixesManager, IMessaging messaging)
    {
        WrappedWorkflow = wrappedWorkflow;
        _fixesManager = fixesManager;
        _messaging = messaging;
        if (WrappedWorkflow.ConflictSearcher != null)
            ConflictSearcherWrapper = new ConflictSearcherWrapper(WrappedWorkflow.ConflictSearcher, WrappedWorkflow,
                _fixesManager, _messaging);
    }

    public bool Execute(IProgressIndicator progressIndicator)
    {
        Logging.LogCallerData();
        var fixedIssues = 0;
        ConflictSearchResult conflictSearchResult = null;
        if (ConflictSearcherWrapper != null)
        {
            conflictSearchResult = ConflictSearcherWrapper.SearchConflictsWithoutConsequences();
            fixedIssues += _fixesManager.RunPreFixes(WrappedWorkflow, conflictSearchResult);
            conflictSearchResult = ConflictSearcherWrapper.SearchConflictsWithoutConsequences();
        }

        var result = WrappedWorkflow.Execute(progressIndicator);

        if (result && conflictSearchResult != null)
            fixedIssues += _fixesManager.RunPostFixes(WrappedWorkflow, conflictSearchResult);
        if (result && fixedIssues > 0)
            _messaging.ShowFixedNotification(MnemonicStore.RemoveMnemonicMark(Title), fixedIssues);
        return result;
    }

    #region Pass-through properties

    public string Title => WrappedWorkflow.Title;

    public IconId Icon => WrappedWorkflow.Icon;

    public string ActionId => WrappedWorkflow.ActionId;

    public string ShortActionId => WrappedWorkflow.ShortActionId;

    public RefactoringActionGroup ActionGroup => WrappedWorkflow.ActionGroup;

    public ISolution Solution => WrappedWorkflow.Solution;

    public IRefactoringPage FirstPendingRefactoringPage => WrappedWorkflow.FirstPendingRefactoringPage;

    public bool MightModifyManyDocuments => WrappedWorkflow.MightModifyManyDocuments;

    public bool HasUI => WrappedWorkflow.HasUI;

    public string HelpKeyword => WrappedWorkflow.HelpKeyword;

    public IEventBus EventBus
    {
        get => WrappedWorkflow.EventBus;
        set => WrappedWorkflow.EventBus = value;
    }

    public Lifetime WorkflowExecuterLifetime
    {
        get => WrappedWorkflow.WorkflowExecuterLifetime;
        set => WrappedWorkflow.WorkflowExecuterLifetime = value;
    }

    public bool RequiresSolutionTransaction => WrappedWorkflow.RequiresSolutionTransaction;

    public IUIApplication Environment => WrappedWorkflow.Environment;

    #endregion Pass-through properties

    #region Pass-through methods

    public override int GetHashCode() => WrappedWorkflow.GetHashCode();

    public bool IsAvailable(IDataContext context) =>
        Logging.LogWrapped(() => WrappedWorkflow.IsAvailable(context));

    public bool Initialize(IDataContext context) =>
        Logging.LogWrapped(() => WrappedWorkflow.Initialize(context));

    public bool PreExecute(IProgressIndicator progressIndicator) =>
        Logging.LogWrapped(() => WrappedWorkflow.PreExecute(progressIndicator));

    public bool PostExecute(IProgressIndicator pi) =>
        Logging.LogWrapped(() => WrappedWorkflow.PostExecute(pi));

    public void SuccessfulFinish(IProgressIndicator pi) =>
        Logging.LogWrapped(() => WrappedWorkflow.SuccessfulFinish(pi));

    public void UnsuccessfulFinish(IProgressIndicator pi) =>
        Logging.LogWrapped(() => WrappedWorkflow.UnsuccessfulFinish(pi));

    public bool RefreshData() =>
        Logging.LogWrapped(WrappedWorkflow.RefreshData);

    public TComponent GetComponent<TComponent>() where TComponent : class =>
        Logging.LogWrapped(WrappedWorkflow.GetComponent<TComponent>);

    public TComponent TryGetComponent<TComponent>() where TComponent : class =>
        Logging.LogWrapped(WrappedWorkflow.TryGetComponent<TComponent>);

    #endregion Pass-through methods

    #region Pass-through events

    public event EventHandler SuccessfullyFinished
    {
        add => WrappedWorkflow.SuccessfullyFinished += value;
        remove => WrappedWorkflow.SuccessfullyFinished -= value;
    }

    public event EventHandler UnsuccessfullyFinished
    {
        add => WrappedWorkflow.UnsuccessfullyFinished += value;
        remove => WrappedWorkflow.UnsuccessfullyFinished -= value;
    }

    #endregion Pass-through events
}
