namespace FakeItEasy.Specs
{
    using System;
    using FakeItEasy;
    using FakeItEasy.Tests.TestHelpers;
    using FluentAssertions;
    using Xbehave;
    using Xunit;

    public static class AssertingCallCountSpecs
    {
        public interface IFoo
        {
            void Method();
        }

        [Scenario]
        public static void MustHaveHappenedNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened()));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it at least once but no calls were made to the fake object.

"));
        }

        [Scenario]
        public static void MustHaveHappened1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened()));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened4Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 4 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened()));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustNotHaveHappenedNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must not have happened"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustNotHaveHappened()));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustNotHaveHappened1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must not have happened"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustNotHaveHappened()));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it never but found it once among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()

"));
        }

        [Scenario]
        public static void MustNotHaveHappened3Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 3 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must not have happened"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustNotHaveHappened()));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it never but found it 3 times among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method() 3 times
    ..."));
        }

        [Scenario]
        public static void MustHaveHappenedOnceExactlyNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened once exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedOnceExactly()));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it exactly once but no calls were made to the fake object.

"));
        }

        [Scenario]
        public static void MustHaveHappenedOnceExactly1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened once exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedOnceExactly()));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappenedOnceExactly2Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 2 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened once exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedOnceExactly()));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it exactly once but found it twice among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method() 2 times
    ..."));
        }

        [Scenario]
        public static void MustHaveHappenedOnceOrMoreNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened once or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedOnceOrMore()));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it at least once but no calls were made to the fake object.

"));
        }

        [Scenario]
        public static void MustHaveHappenedOnceOrMore1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened once or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedOnceOrMore()));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappenedOnceOrMore2Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 2 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened once or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedOnceOrMore()));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappenedOnceOrLessNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened once or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedOnceOrLess()));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappenedOnceOrLess1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened once or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedOnceOrLess()));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappenedOnceOrLess2Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 2 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened once or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedOnceOrLess()));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it no more than once but found it twice among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method() 2 times
    ...

"));
        }

        [Scenario]
        public static void MustHaveHappenedTwiceExactlyNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened twice exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedTwiceExactly()));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it exactly twice but no calls were made to the fake object.

"));
        }

        [Scenario]
        public static void MustHaveHappenedTwiceExactly1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened twice exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedTwiceExactly()));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it exactly twice but found it once among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()

"));
        }

        [Scenario]
        public static void MustHaveHappenedTwiceExactly2Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 2 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened twice exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedTwiceExactly()));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappenedTwiceExactly3Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 3 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened twice exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedTwiceExactly()));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it exactly twice but found it 3 times among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method() 3 times
    ..."));
        }

        [Scenario]
        public static void MustHaveHappenedTwiceOrMoreNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened twice or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedTwiceOrMore()));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it at least twice but no calls were made to the fake object.

"));
        }

        [Scenario]
        public static void MustHaveHappenedTwiceOrMore1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened twice or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedTwiceOrMore()));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it at least twice but found it once among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()

"));
        }

        [Scenario]
        public static void MustHaveHappenedTwiceOrMore2Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 2 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened twice or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedTwiceOrMore()));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappenedTwiceOrMore3Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 3 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened twice or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedTwiceOrMore()));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappenedTwiceOrLessNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened twice or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedTwiceOrLess()));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappenedTwiceOrLess1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened twice or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedTwiceOrLess()));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappenedTwiceOrLess2Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 2 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened twice or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedTwiceOrLess()));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappenedTwiceOrLess3Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 3 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened twice or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedTwiceOrLess()));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it no more than twice but found it 3 times among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method() 3 times
    ...

"));
        }

        [Scenario]
        public static void MustHaveHappened0TimesExactlyNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened 0 times exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(0, Times.Exactly)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened0TimesExactly1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened 0 times exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(0, Times.Exactly)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it exactly 0 times but found it once among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
"));
        }

        [Scenario]
        public static void MustHaveHappened0TimesOrMoreNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened 0 times or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(0, Times.OrMore)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened0TimesOrMore1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened 0 times or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(0, Times.OrMore)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened0TimesOrLessNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened 0 times or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(0, Times.OrLess)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened0TimesOrLess1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() =>
                {
                    fake.Method();
                });

            "When I assert that the call must have happened 0 times or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(0, Times.OrLess)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it no more than 0 times but found it once among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()

"));
        }

        [Scenario]
        public static void MustHaveHappened1TimesExactlyNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened 1 times exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(1, Times.Exactly)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it exactly 1 times but no calls were made to the fake object.

"));
        }

        [Scenario]
        public static void MustHaveHappened1TimesExactly1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened 1 times exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(1, Times.Exactly)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened1TimesExactly2Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 2 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened 1 times exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(1, Times.Exactly)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it exactly 1 times but found it twice among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method() 2 times
    ...

"));
        }

        [Scenario]
        public static void MustHaveHappened1TimesOrMoreNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened 1 times or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(1, Times.OrMore)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it at least 1 times but no calls were made to the fake object.

"));
        }

        [Scenario]
        public static void MustHaveHappened1TimesOrMore1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened 1 times or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(1, Times.OrMore)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened1TimesOrMore2Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 2 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened 1 times or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(1, Times.OrMore)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened1TimesOrLessNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened 1 times or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(1, Times.OrLess)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened1TimesOrLess1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened 1 times or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(1, Times.OrLess)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened1TimesOrLess2Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 2 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened 1 times or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(1, Times.OrLess)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it no more than 1 times but found it twice among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method() 2 times
    ...

"));
        }

        [Scenario]
        public static void MustHaveHappened2TimesExactlyNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened 2 times exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(2, Times.Exactly)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it exactly 2 times but no calls were made to the fake object.

"));
        }

        [Scenario]
        public static void MustHaveHappened2TimesExactly1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened 2 times exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(2, Times.Exactly)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it exactly 2 times but found it once among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()

"));
        }

        [Scenario]
        public static void MustHaveHappened2TimesExactly2Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 2 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened 2 times exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(2, Times.Exactly)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened2TimesExactly3Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 3 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened 2 times exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(2, Times.Exactly)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it exactly 2 times but found it 3 times among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method() 3 times
    ...

"));
        }

        [Scenario]
        public static void MustHaveHappened2TimesOrMoreNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened 2 times or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(2, Times.OrMore)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it at least 2 times but no calls were made to the fake object.

"));
        }

        [Scenario]
        public static void MustHaveHappened2TimesOrMore1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened 2 times or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(2, Times.OrMore)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it at least 2 times but found it once among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()

"));
        }

        [Scenario]
        public static void MustHaveHappened2TimesOrMore2Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 2 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened 2 times or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(2, Times.OrMore)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened2TimesOrMore3Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 3 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened 2 times or more"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(2, Times.OrMore)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened2TimesOrLessNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened 2 times or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(2, Times.OrLess)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened2TimesOrLess1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened 2 times or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(2, Times.OrLess)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened2TimesOrLess2Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 2 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened 2 times or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(2, Times.OrLess)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened2TimesOrLess3Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 3 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened 2 times or less"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(2, Times.OrLess)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it no more than 2 times but found it 3 times among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method() 3 times
    ...

"));
        }

        [Scenario]
        public static void MustHaveHappened3TimesExactlyNoCalls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make no calls to a faked method"
                .x(() => { });

            "When I assert that the call must have happened 3 times exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(3, Times.Exactly)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it exactly 3 times but no calls were made to the fake object.

"));
        }

        [Scenario]
        public static void MustHaveHappened3TimesExactly1Call(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 1 call to a faked method"
                .x(() => fake.Method());

            "When I assert that the call must have happened 3 times exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(3, Times.Exactly)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it exactly 3 times but found it once among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()

"));
        }

        [Scenario]
        public static void MustHaveHappened3TimesExactly3Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 3 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened 3 times exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(3, Times.Exactly)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }

        [Scenario]
        public static void MustHaveHappened3TimesExactly4Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 4 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened 3 times exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(3, Times.Exactly)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it exactly 3 times but found it 4 times among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method() 4 times
    ..."));
        }

        [Scenario]
        public static void MustHaveHappened3TimesOrMore2Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 2 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened 3 times exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(3, Times.OrMore)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it at least 3 times but found it twice among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method() 2 times
    ..."));
        }

        [Scenario]
        public static void MustHaveHappened3TimesOrLess4Calls(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 4 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened 3 times exactly"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappened(3, Times.OrLess)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it no more than 3 times but found it 4 times among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method() 4 times
    ..."));
        }

        [Scenario]
        public static void MustHaveHappenedANumberOfTimesMatchingExpressionNoMatch(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 3 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened an even number of times"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedANumberOfTimesMatching(n => n % 2 == 0)));

            "Then the assertion fails"
                .x(() => exception.Should()
                    .BeAnExceptionOfType<ExpectationException>()
                    .WithMessage(@"

  Assertion failed for the following call:
    FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method()
  Expected to find it the number of times specified by the predicate 'n => ((n % 2) == 0)' but found it 3 times among the calls:
    1: FakeItEasy.Specs.AssertingCallCountSpecs+IFoo.Method() 3 times
    ...

"));
        }

        [Scenario]
        public static void MustHaveHappenedANumberOfTimesMatchingInlineExpressionMatch(IFoo fake, Exception exception)
        {
            "Given a Fake"
                .x(() => fake = A.Fake<IFoo>());

            "And I make 4 calls to a faked method"
                .x(() =>
                {
                    fake.Method();
                    fake.Method();
                    fake.Method();
                    fake.Method();
                });

            "When I assert that the call must have happened an even number of times"
                .x(() => exception = Record.Exception(() => A.CallTo(() => fake.Method()).MustHaveHappenedANumberOfTimesMatching(n => n % 2 == 0)));

            "Then the assertion succeeds"
                .x(() => exception.Should().BeNull());
        }
    }
}
